# Hello, BIOS

## What is the BIOS?

BIOS stands for Basic Input/Output System, and it's the first
program that runs when you start your computer. You can't access
the code for the BIOS, since it's usually stored in Read-Only
memory on the motherboard (which is also why it's such a pain to
do BIOS updates). This program is responsible for the Power-On
Self Test (POST) and makes sure you have a correct and operational
system configuration.

One more thing you need to know: the BIOS doesn't load the boot
sector at address 0. That's reserved for other things to use. Instead,
our code gets loaded at address `0x7C00`. This wasn't an issue for our
simple example since we weren't doing complex jumps, but if we're not
careful we'll jump to the wrong address, which would cause QEMU to hang
or crash altogether. Luckily, NASM makes this easy for us! We just need
to add `[org 0x7C00]` to the beginning of our file to tell NASM that's 
where our code is loaded.

## Bios Utilities

When the computer first starts, only the first sector (512 bytes) has
been loaded from the disk. Obviously, this is not enough to store an
entire bootloader, or even enough code for the hard disk driver. So
to help the developers out, the creators of the BIOS gave us access to
a few of the BIOS utilities to use. The two most important ones (and the
ones we'll be using for this project) are the 'print character' and 'load
sector' utilities. The first one allows us to log messages, while the
second allows us to load additional code from the disk! This comes in handy
if you need more than 512 bytes for your boot sector (which nearly everyone
does).

For a more complete list of the BIOS functions, please see this [archive
page](https://web.archive.org/web/20200921172844/https://wiki.osdev.org/BIOS).
This archive is recent as of September 21, 2020.

## Defining a String

There are several ways to define a string character, but for the purposes of
this tutorial we'll be using a C-Style null-terminated string. In assembly,
we'll use the `db` command to define our characters, and encapsulate the string
with backticks (`). In NASM, the backtick allows us to use C-Style escapes for
carriage return and newline ('\r' and '\n'), while the more familiar quotes (' 
or ") would define the characters as they appear.

At the end of the string, we'll need to be sure to add a terminating null
character, so we close the string with a second backtick and then add `, 0`.
This adds a final zero-byte to the end of the string, which allows us to know
when our string is complete while we're printing it. Take a look at the `boot.asm`
file for a complete example.

## Printing to the Console

BIOS Utilities are usually called using a **CPU Interrupt**. These are usually
generated by the processor itself or through external devices, but you
can also generate them yourself using assembly. The `int` command causes
the CPU to self-interrupt with a given command code, which is usually a
single byte.

The BIOS uses interrupts `0x10` to `0x16` inclusive, and each interrupt has
a different function. In this case, we want to use the screen interrupt, which
just so happens to be `0x10`. If you take a look at this 
[article about the BIOS](https://web.archive.org/web/20200921172844/https://wiki.osdev.org/BIOS)
you'll see that the interrupt `0x10` can do a number of different tasks depending
on the value of the register `AX`. In our case, we want to print a character to
the screen, so we need to make sure that the value of `AH` is the value `0x0E`,
and the value in `AL` is the character we want to print.

To print our string to the console, we'll want to use a loop, printing each individual
character to the console, and jump to the end once we reach the terminating
null character. This would be a good time to look at the `print.asm` file and take
a look at the source code.

## Including External Files

We can use the `%include` directive in NASM to copy code files verbatim into
our program. We use this pretty extensively to make sure that our assembly files
don't get too jumbled up. This is why we included `print.asm` in the source, but
only need to assemble `boot.asm`, it's already been included!

## Building

Building this example is the same as building the previous step. To
assemble the program, use:

```sh
nasm boot.asm
```

Then, to run the program in QEMU, use the following command:

```sh
qemu-system-x86_64 -drive format=raw,file=boot
```

## Try it Yourself

One of the things we'll need for the next chapter is a way to print numbers
to the console. Try creating your own `print_hex` function which prints a
16-bit number to the console in hexadecimal. Don't forget the `0x` prefix 
when you print your number to the screen!